/*
 * AttackInfo.h
 *
 * Created 9/6/2009 By Johnny Huynh
 *
 * Version 00.00.02 9/6/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef ATTACK_INFO_H
 #define ATTACK_INFO_H
 
 template <typename T> class AttackInfo;
 
 #include "global.h"
 
 #include "ActionInfo.h"
 #include "DamageObject.h"
 #include "pointerTo.h"
 
 /**
  * Class specification for AttackInfo
  */
 template <typename T>
 class AttackInfo : virtual public ActionInfo<T>
 {
 // Protected Static Functions
 protected:
    static inline void upon_death_handler( GenericAsyncTask* task_Ptr, bool clean_exit, void* data_Ptr );
    
 // Data Members
 private:
    PT(DamageObject<T>) _dmg_obj_Ptr;
    double _duration_before_attaching_dmg_obj;
    double _duration_before_detaching_dmg_obj; //_duration_before_detaching_dmg_obj_after_attachment;
    
 // Local Functions
 public:
    AttackInfo( DamageObject<T>* dmg_obj_Ptr = NULL, 
                const double duration_before_attaching_dmg_obj = 0.0, 
                const double duration_before_detaching_dmg_obj = 0.0 );
    AttackInfo( const AttackInfo<T>& attack_info );
    virtual ~AttackInfo();
    inline AttackInfo<T>& operator=( const AttackInfo<T>& attack_info );
    inline DamageObject<T>* get_damage_object();
    inline double get_duration_before_attaching_damage_object() const;
    inline double get_duration_before_detaching_damage_object() const;
    virtual inline void split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr ) = 0;
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** PROTECTED STATIC FUNCTIONS **/
 
 /**
  * upon_death_handler() detaches the DamageObject that may have originally been attached.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask. The ActionInfo
  * of the ActionTask must also be an AttackInfo.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (bool) clean_exit
  * @param (void*) data_Ptr
  */
 template <typename T>
 inline void AttackInfo<T>::upon_death_handler( GenericAsyncTask* task_Ptr, bool clean_exit, void* data_Ptr )
 {
    nassertv( task_Ptr != NULL );
    
    PT(ActionTask<T>) action_task_Ptr( reinterpret_cast<ActionTask<T>*>( task_Ptr ) );
    PT(AttackInfo<T>) attack_info_Ptr( dynamic_cast<AttackInfo<T>*>( action_task_Ptr->get_action_info() ) );
    nassertv( attack_info_Ptr != NULL );
    attack_info_Ptr->get_damage_object()->detach_node();
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 AttackInfo<T>::AttackInfo( DamageObject<T>* dmg_obj_Ptr, 
                            const double duration_before_attaching_dmg_obj, 
                            const double duration_before_detaching_dmg_obj )
               : ActionInfo<T>(),
                 _dmg_obj_Ptr( dmg_obj_Ptr ), 
                 _duration_before_attaching_dmg_obj( duration_before_attaching_dmg_obj ),
                 _duration_before_detaching_dmg_obj( duration_before_detaching_dmg_obj )
 {
    nassertv( _dmg_obj_Ptr != NULL );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 AttackInfo<T>::AttackInfo( const AttackInfo<T>& attack_info )
               : ActionInfo<T>( attack_info ),
                 _dmg_obj_Ptr( attack_info._dmg_obj_Ptr ), 
                 _duration_before_attaching_dmg_obj( attack_info._duration_before_attaching_dmg_obj ),
                 _duration_before_detaching_dmg_obj( attack_info._duration_before_detaching_dmg_obj )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 AttackInfo<T>::~AttackInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified AttackInfo to this AttackInfo.
  *
  * @param (const AttackInfo<T>&) attack_info
  * @return AttackInfo<T>&
  */
 template <typename T>
 inline AttackInfo<T>& AttackInfo<T>::operator=( const AttackInfo<T>& attack_info )
 {
    ActionInfo<T>::operator=( attack_info );
    _dmg_obj_Ptr = attack_info._dmg_obj_Ptr; 
    _duration_before_attaching_dmg_obj = attack_info._duration_before_attaching_dmg_obj;
    _duration_before_detaching_dmg_obj = attack_info._duration_before_detaching_dmg_obj;
    
    return *this;
 }
 
 /**
  * get_damage_object() returns a pointer to the DamageObject of this AttackInfo.
  *
  * @return DamageObject<T>*
  */
 template <typename T>
 inline DamageObject<T>* AttackInfo<T>::get_damage_object()
 {
    return _dmg_obj_Ptr;
 }
 
 /**
  * get_duration_before_attaching_damage_object() returns the duration in the global clock
  * time before the DamageObject of this AttackInfo is attached to a parent.
  *
  * @return double
  */
 template <typename T>
 inline double AttackInfo<T>::get_duration_before_attaching_damage_object() const
 {
    return _duration_before_attaching_dmg_obj;
 }
 
 /**
  * get_duration_before_detaching_damage_object() returns the duration in the global clock
  * time before the DamageObject of this AttackInfo is detached from its parent.
  *
  * @return double
  */
 template <typename T>
 inline double AttackInfo<T>::get_duration_before_detaching_damage_object() const
 {
    return _duration_before_detaching_dmg_obj;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // ATTACK_INFO_H